import { ConfirmationDialog } from '@/components/confirmation-dialog';
import { Button } from '@/components/ui/button';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Separator } from '@/components/ui/separator';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs';
import AppLayout from '@/layouts/app-layout';
import { Head, router, useForm } from '@inertiajs/react';
import { Key, Trash2, User } from 'lucide-react';
import { FormEvent, useState } from 'react';
import { toast } from 'sonner';

interface Props {
    user: {
        id: number;
        name: string;
        email: string;
        role: string;
    };
}

export default function SettingsIndex({ user }: Props) {
    const [activeTab, setActiveTab] = useState('profile');

    // Profile form
    const profileForm = useForm({
        name: user.name,
        email: user.email,
    });

    // Password form
    const passwordForm = useForm({
        current_password: '',
        password: '',
        password_confirmation: '',
    });

    // Delete account form
    const [showDeleteConfirm, setShowDeleteConfirm] = useState(false);
    const [deletePassword, setDeletePassword] = useState('');
    const [deleteDialog, setDeleteDialog] = useState(false);

    const handleProfileUpdate = (e: FormEvent) => {
        e.preventDefault();
        router.put('/settings/profile', profileForm.data, {
            onSuccess: () => {
                toast.success('Profile updated successfully!');
            },
            onError: () => {
                toast.error('Failed to update profile');
            },
        });
    };

    const handlePasswordUpdate = (e: FormEvent) => {
        e.preventDefault();
        router.put('/settings/password', passwordForm.data, {
            onSuccess: () => {
                toast.success('Password updated successfully!');
                passwordForm.reset();
            },
            onError: () => {
                toast.error('Failed to update password');
            },
        });
    };

    const handleDeleteClick = () => {
        if (!deletePassword) {
            toast.error('Please enter your password');
            return;
        }
        setDeleteDialog(true);
    };

    const handleDeleteAccount = () => {
        router.delete('/settings/profile', {
            data: { password: deletePassword },
            onSuccess: () => {
                toast.success('Account deleted successfully');
            },
            onError: () => {
                toast.error('Failed to delete account');
            },
        });
    };

    return (
        <AppLayout>
            <Head title="Settings" />

            <div className="container mx-auto px-4 py-8 max-w-4xl">
                <div className="mb-8">
                    <h1 className="text-4xl font-bold mb-2">Settings</h1>
                    <p className="text-muted-foreground">
                        Manage your account settings and preferences
                    </p>
                </div>

                <Tabs value={activeTab} onValueChange={setActiveTab}>
                    <TabsList className="grid w-full grid-cols-3">
                        <TabsTrigger value="profile">
                            <User className="mr-2 h-4 w-4" />
                            Profile
                        </TabsTrigger>
                        <TabsTrigger value="password">
                            <Key className="mr-2 h-4 w-4" />
                            Password
                        </TabsTrigger>
                        <TabsTrigger value="account">
                            <Trash2 className="mr-2 h-4 w-4" />
                            Account
                        </TabsTrigger>
                    </TabsList>

                    {/* Profile Tab */}
                    <TabsContent value="profile">
                        <Card>
                            <CardHeader>
                                <CardTitle>Profile Information</CardTitle>
                                <CardDescription>
                                    Update your account's profile information and email address
                                </CardDescription>
                            </CardHeader>
                            <CardContent>
                                <form onSubmit={handleProfileUpdate} className="space-y-4">
                                    <div className="space-y-2">
                                        <Label htmlFor="name">Name</Label>
                                        <Input
                                            id="name"
                                            value={profileForm.data.name}
                                            onChange={(e) => profileForm.setData('name', e.target.value)}
                                            required
                                        />
                                        {profileForm.errors.name && (
                                            <p className="text-sm text-red-500">{profileForm.errors.name}</p>
                                        )}
                                    </div>

                                    <div className="space-y-2">
                                        <Label htmlFor="email">Email</Label>
                                        <Input
                                            id="email"
                                            type="email"
                                            value={profileForm.data.email}
                                            onChange={(e) => profileForm.setData('email', e.target.value)}
                                            required
                                        />
                                        {profileForm.errors.email && (
                                            <p className="text-sm text-red-500">{profileForm.errors.email}</p>
                                        )}
                                    </div>

                                    <div className="space-y-2">
                                        <Label>Role</Label>
                                        <Input value={user.role} disabled className="capitalize" />
                                        <p className="text-sm text-muted-foreground">
                                            Contact an administrator to change your role
                                        </p>
                                    </div>

                                    <Button type="submit" disabled={profileForm.processing}>
                                        {profileForm.processing ? 'Saving...' : 'Save Changes'}
                                    </Button>
                                </form>
                            </CardContent>
                        </Card>
                    </TabsContent>

                    {/* Password Tab */}
                    <TabsContent value="password">
                        <Card>
                            <CardHeader>
                                <CardTitle>Update Password</CardTitle>
                                <CardDescription>
                                    Ensure your account is using a long, random password to stay secure
                                </CardDescription>
                            </CardHeader>
                            <CardContent>
                                <form onSubmit={handlePasswordUpdate} className="space-y-4">
                                    <div className="space-y-2">
                                        <Label htmlFor="current_password">Current Password</Label>
                                        <Input
                                            id="current_password"
                                            type="password"
                                            value={passwordForm.data.current_password}
                                            onChange={(e) => passwordForm.setData('current_password', e.target.value)}
                                            required
                                        />
                                        {passwordForm.errors.current_password && (
                                            <p className="text-sm text-red-500">{passwordForm.errors.current_password}</p>
                                        )}
                                    </div>

                                    <div className="space-y-2">
                                        <Label htmlFor="password">New Password</Label>
                                        <Input
                                            id="password"
                                            type="password"
                                            value={passwordForm.data.password}
                                            onChange={(e) => passwordForm.setData('password', e.target.value)}
                                            required
                                        />
                                        {passwordForm.errors.password && (
                                            <p className="text-sm text-red-500">{passwordForm.errors.password}</p>
                                        )}
                                    </div>

                                    <div className="space-y-2">
                                        <Label htmlFor="password_confirmation">Confirm Password</Label>
                                        <Input
                                            id="password_confirmation"
                                            type="password"
                                            value={passwordForm.data.password_confirmation}
                                            onChange={(e) => passwordForm.setData('password_confirmation', e.target.value)}
                                            required
                                        />
                                    </div>

                                    <Button type="submit" disabled={passwordForm.processing}>
                                        {passwordForm.processing ? 'Updating...' : 'Update Password'}
                                    </Button>
                                </form>
                            </CardContent>
                        </Card>
                    </TabsContent>

                    {/* Account Tab */}
                    <TabsContent value="account">
                        <Card className="border-red-200">
                            <CardHeader>
                                <CardTitle className="text-red-600">Delete Account</CardTitle>
                                <CardDescription>
                                    Permanently delete your account and all associated data
                                </CardDescription>
                            </CardHeader>
                            <CardContent className="space-y-4">
                                <div className="rounded-lg bg-red-50 p-4 dark:bg-red-950">
                                    <h4 className="font-semibold text-red-800 dark:text-red-200 mb-2">
                                        Warning: This action is irreversible
                                    </h4>
                                    <ul className="list-disc list-inside space-y-1 text-sm text-red-700 dark:text-red-300">
                                        <li>All your sections will be deleted</li>
                                        <li>Your profile information will be permanently removed</li>
                                        <li>You will be immediately logged out</li>
                                        <li>This action cannot be undone</li>
                                    </ul>
                                </div>

                                <Separator />

                                {!showDeleteConfirm ? (
                                    <Button
                                        variant="destructive"
                                        onClick={() => setShowDeleteConfirm(true)}
                                    >
                                        <Trash2 className="mr-2 h-4 w-4" />
                                        Delete My Account
                                    </Button>
                                ) : (
                                    <div className="space-y-4">
                                        <div className="space-y-2">
                                            <Label htmlFor="delete_password">
                                                Confirm your password to delete your account
                                            </Label>
                                            <Input
                                                id="delete_password"
                                                type="password"
                                                value={deletePassword}
                                                onChange={(e) => setDeletePassword(e.target.value)}
                                                placeholder="Enter your password"
                                            />
                                        </div>

                                        <div className="flex gap-2">
                                            <Button
                                                variant="outline"
                                                onClick={() => {
                                                    setShowDeleteConfirm(false);
                                                    setDeletePassword('');
                                                }}
                                            >
                                                Cancel
                                            </Button>
                                            <Button
                                                variant="destructive"
                                                onClick={handleDeleteClick}
                                                disabled={!deletePassword}
                                            >
                                                Yes, Delete My Account
                                            </Button>
                                        </div>
                                    </div>
                                )}
                            </CardContent>
                        </Card>
                    </TabsContent>
                </Tabs>
            </div>

            <ConfirmationDialog
                open={deleteDialog}
                onOpenChange={setDeleteDialog}
                onConfirm={handleDeleteAccount}
                title="Delete Account"
                description="Are you absolutely sure? This action cannot be undone. All your data will be permanently deleted."
                confirmText="Yes, Delete My Account"
                cancelText="Cancel"
                variant="destructive"
            />
        </AppLayout>
    );
}
