import { Button } from '@/components/ui/button';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import {
    Select,
    SelectContent,
    SelectItem,
    SelectTrigger,
    SelectValue,
} from '@/components/ui/select';
import { Textarea } from '@/components/ui/textarea';
import AppLayout from '@/layouts/app-layout';
import type { Category } from '@/types/section';
import { Head, router, useForm, usePage } from '@inertiajs/react';
import { ArrowLeft, Plus, Trash2 } from 'lucide-react';
import { FormEvent, useEffect, useState } from 'react';
import { toast } from 'sonner';

interface Props {
    categories: Category[];
}

interface Version {
    framework: string;
    styling: string;
    code: string;
    dependencies: string;
    notes: string;
}

export default function CreateSection({ categories }: Props) {
    const { props } = usePage<{ success?: string; error?: string }>();
    const { data, setData, processing, errors } = useForm({
        name: '',
        description: '',
        category_id: '',
        tags: '',
        thumbnail: null as File | null,
    });

    const [versions, setVersions] = useState<Version[]>([
        {
            framework: 'html',
            styling: 'tailwind',
            code: '',
            dependencies: '',
            notes: '',
        },
    ]);

    // Show flash messages
    useEffect(() => {
        if (props.success) {
            toast.success(props.success);
        }
        if (props.error) {
            toast.error(props.error);
        }
        // Show validation errors for versions
        if (props.errors?.versions) {
            toast.error(props.errors.versions);
        }
    }, [props.success, props.error, props.errors]);

    const addVersion = () => {
        setVersions([
            ...versions,
            {
                framework: 'html',
                styling: 'tailwind',
                code: '',
                dependencies: '',
                notes: '',
            },
        ]);
    };

    const removeVersion = (index: number) => {
        if (versions.length > 1) {
            setVersions(versions.filter((_, i) => i !== index));
        }
    };

    const updateVersion = (index: number, field: keyof Version, value: string) => {
        const newVersions = [...versions];
        newVersions[index] = { ...newVersions[index], [field]: value };
        setVersions(newVersions);
    };

    const handleSubmit = (e: FormEvent) => {
        e.preventDefault();

        // Convert tags string to array
        const tagsArray = data.tags.split(',').map(tag => tag.trim()).filter(tag => tag);

        // Prepare versions data
        const versionsData = versions.map(version => ({
            framework: version.framework,
            styling: version.styling,
            code: version.code,
            dependencies: version.dependencies.split(',').map(dep => dep.trim()).filter(dep => dep),
            notes: version.notes,
        }));

        router.post('/sections', {
            name: data.name,
            description: data.description,
            category_id: data.category_id,
            tags: tagsArray,
            thumbnail: data.thumbnail,
            versions: versionsData,
        });
    };

    return (
        <AppLayout>
            <Head title="Create Section" />

            <div className="container mx-auto px-4 py-8 max-w-4xl">
                {/* Header */}
                <div className="mb-8">
                    <Button
                        variant="ghost"
                        className="mb-4"
                        onClick={() => window.history.back()}
                    >
                        <ArrowLeft className="mr-2 h-4 w-4" />
                        Back
                    </Button>
                    <h1 className="text-4xl font-bold mb-2">Create New Section</h1>
                    <p className="text-muted-foreground">
                        Add a new component to the library
                    </p>
                </div>

                <form onSubmit={handleSubmit} className="space-y-6">
                    {/* Basic Information */}
                    <Card>
                        <CardHeader>
                            <CardTitle>Basic Information</CardTitle>
                            <CardDescription>
                                Provide the basic details about your component
                            </CardDescription>
                        </CardHeader>
                        <CardContent className="space-y-4">
                            {/* Name */}
                            <div className="space-y-2">
                                <Label htmlFor="name">
                                    Name <span className="text-red-500">*</span>
                                </Label>
                                <Input
                                    id="name"
                                    value={data.name}
                                    onChange={(e) => setData('name', e.target.value)}
                                    placeholder="e.g., Modern Hero Section"
                                    required
                                />
                                {errors.name && (
                                    <p className="text-sm text-red-500">{errors.name}</p>
                                )}
                            </div>

                            {/* Description */}
                            <div className="space-y-2">
                                <Label htmlFor="description">Description</Label>
                                <Textarea
                                    id="description"
                                    value={data.description}
                                    onChange={(e) => setData('description', e.target.value)}
                                    placeholder="Describe your component..."
                                    rows={3}
                                />
                                {errors.description && (
                                    <p className="text-sm text-red-500">{errors.description}</p>
                                )}
                            </div>

                            {/* Category */}
                            <div className="space-y-2">
                                <Label htmlFor="category">
                                    Category <span className="text-red-500">*</span>
                                </Label>
                                <Select
                                    value={data.category_id}
                                    onValueChange={(value) => setData('category_id', value)}
                                    required
                                >
                                    <SelectTrigger>
                                        <SelectValue placeholder="Select a category" />
                                    </SelectTrigger>
                                    <SelectContent>
                                        {categories.map((category) => (
                                            <SelectItem key={category.id} value={category.id.toString()}>
                                                {category.name}
                                            </SelectItem>
                                        ))}
                                    </SelectContent>
                                </Select>
                                {errors.category_id && (
                                    <p className="text-sm text-red-500">{errors.category_id}</p>
                                )}
                            </div>

                            {/* Tags */}
                            <div className="space-y-2">
                                <Label htmlFor="tags">Tags (comma-separated)</Label>
                                <Input
                                    id="tags"
                                    value={data.tags}
                                    onChange={(e) => setData('tags', e.target.value)}
                                    placeholder="e.g., gradient, modern, responsive"
                                />
                                {errors.tags && (
                                    <p className="text-sm text-red-500">{errors.tags}</p>
                                )}
                            </div>

                            {/* Thumbnail */}
                            <div className="space-y-2">
                                <Label htmlFor="thumbnail">Thumbnail Image</Label>
                                <Input
                                    id="thumbnail"
                                    type="file"
                                    accept="image/*"
                                    onChange={(e) => setData('thumbnail', e.target.files?.[0] || null)}
                                />
                                {errors.thumbnail && (
                                    <p className="text-sm text-red-500">{errors.thumbnail}</p>
                                )}
                            </div>
                        </CardContent>
                    </Card>

                    {/* Versions */}
                    <div className="space-y-4">
                        <div className="flex items-center justify-between">
                            <div>
                                <h2 className="text-2xl font-bold">Code Versions</h2>
                                <p className="text-sm text-muted-foreground">
                                    Add code for different frameworks and styling options
                                </p>
                            </div>
                            <Button type="button" onClick={addVersion} variant="outline">
                                <Plus className="mr-2 h-4 w-4" />
                                Add Version
                            </Button>
                        </div>

                        {versions.map((version, index) => (
                            <Card key={index}>
                                <CardHeader>
                                    <div className="flex items-center justify-between">
                                        <CardTitle className="text-lg">
                                            Version {index + 1}
                                        </CardTitle>
                                        {versions.length > 1 && (
                                            <Button
                                                type="button"
                                                variant="destructive"
                                                size="sm"
                                                onClick={() => removeVersion(index)}
                                            >
                                                <Trash2 className="h-4 w-4" />
                                            </Button>
                                        )}
                                    </div>
                                </CardHeader>
                                <CardContent className="space-y-4">
                                    <div className="grid grid-cols-2 gap-4">
                                        {/* Framework */}
                                        <div className="space-y-2">
                                            <Label>Framework</Label>
                                            <Select
                                                value={version.framework}
                                                onValueChange={(value) =>
                                                    updateVersion(index, 'framework', value)
                                                }
                                            >
                                                <SelectTrigger>
                                                    <SelectValue />
                                                </SelectTrigger>
                                                <SelectContent>
                                                    <SelectItem value="html">HTML</SelectItem>
                                                    <SelectItem value="react">React</SelectItem>
                                                    <SelectItem value="vue">Vue</SelectItem>
                                                    <SelectItem value="angular">Angular</SelectItem>
                                                </SelectContent>
                                            </Select>
                                        </div>

                                        {/* Styling */}
                                        <div className="space-y-2">
                                            <Label>Styling</Label>
                                            <Select
                                                value={version.styling}
                                                onValueChange={(value) =>
                                                    updateVersion(index, 'styling', value)
                                                }
                                            >
                                                <SelectTrigger>
                                                    <SelectValue />
                                                </SelectTrigger>
                                                <SelectContent>
                                                    <SelectItem value="tailwind">Tailwind CSS</SelectItem>
                                                    <SelectItem value="bootstrap">Bootstrap</SelectItem>
                                                </SelectContent>
                                            </Select>
                                        </div>
                                    </div>

                                    {/* Code */}
                                    <div className="space-y-2">
                                        <Label>Code</Label>
                                        <Textarea
                                            value={version.code}
                                            onChange={(e) =>
                                                updateVersion(index, 'code', e.target.value)
                                            }
                                            placeholder="Paste your code here..."
                                            rows={10}
                                            className="font-mono text-sm"
                                        />
                                    </div>

                                    {/* Dependencies */}
                                    <div className="space-y-2">
                                        <Label>Dependencies (comma-separated)</Label>
                                        <Input
                                            value={version.dependencies}
                                            onChange={(e) =>
                                                updateVersion(index, 'dependencies', e.target.value)
                                            }
                                            placeholder="e.g., react, react-dom"
                                        />
                                    </div>

                                    {/* Notes */}
                                    <div className="space-y-2">
                                        <Label>Notes</Label>
                                        <Textarea
                                            value={version.notes}
                                            onChange={(e) =>
                                                updateVersion(index, 'notes', e.target.value)
                                            }
                                            placeholder="Any additional notes or instructions..."
                                            rows={3}
                                        />
                                    </div>
                                </CardContent>
                            </Card>
                        ))}
                    </div>

                    {/* Submit */}
                    <div className="flex gap-2 justify-end">
                        <Button
                            type="button"
                            variant="outline"
                            onClick={() => window.history.back()}
                        >
                            Cancel
                        </Button>
                        <Button type="submit" disabled={processing}>
                            {processing ? 'Creating...' : 'Create Section'}
                        </Button>
                    </div>
                </form>
            </div>
        </AppLayout>
    );
}
