import { Button } from '@/components/ui/button';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import AppLayout from '@/layouts/app-layout';
import { dashboard } from '@/routes';
import { type BreadcrumbItem } from '@/types';
import { Head, Link, usePage } from '@inertiajs/react';
import {
    BarChart3,
    CheckSquare,
    Clock,
    Copy,
    Eye,
    Package,
    Plus,
    TrendingUp,
    Users
} from 'lucide-react';

const breadcrumbs: BreadcrumbItem[] = [
    {
        title: 'Dashboard',
        href: dashboard().url,
    },
];

interface DashboardStats {
    // Admin stats
    totalSections?: number;
    approvedSections?: number;
    pendingSections?: number;
    rejectedSections?: number;
    totalCategories?: number;
    totalViews?: number;
    totalCopies?: number;
    totalUsers?: number;
    totalContributors?: number;

    // Contributor stats
    mySections?: number;
    myApprovedSections?: number;
    myPendingSections?: number;
    myRejectedSections?: number;
    myCategories?: number;
    myTotalViews?: number;
    myTotalCopies?: number;
}

interface DashboardProps {
    stats?: DashboardStats;
}

export default function Dashboard({ stats }: DashboardProps) {
    const page = usePage();
    const auth = (page.props as unknown as { auth: { user: { role: string; name: string } } }).auth;
    const userRole = auth?.user?.role || 'user';
    const userName = auth?.user?.name || 'User';

    // Default stats if not provided
    const dashboardStats: DashboardStats = stats || {
        totalSections: 0,
        approvedSections: 0,
        pendingSections: 0,
        rejectedSections: 0,
        mySections: 0,
        myPendingSections: 0,
        totalViews: 0,
        totalCopies: 0,
        totalUsers: 0,
    };

    return (
        <AppLayout breadcrumbs={breadcrumbs}>
            <Head title="Dashboard" />

            <div className="flex h-full flex-1 flex-col gap-6 p-6">
                {/* Welcome Section */}
                <div>
                    <h1 className="text-3xl font-bold">Welcome back, {userName}!</h1>
                    <p className="text-muted-foreground mt-1">
                        {userRole === 'admin' && 'Manage your component library and users'}
                        {userRole === 'contributor' && 'Create and manage your UI components'}
                        {userRole === 'user' && 'Browse and use thousands of UI components'}
                    </p>
                </div>

                {/* Stats Grid */}
                <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-4">
                    {/* For Contributors - Show their own stats */}
                    {userRole === 'contributor' && (
                        <>
                            <Card>
                                <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                                    <CardTitle className="text-sm font-medium">My Sections</CardTitle>
                                    <Package className="h-4 w-4 text-muted-foreground" />
                                </CardHeader>
                                <CardContent>
                                    <div className="text-2xl font-bold">{dashboardStats.mySections}</div>
                                    <p className="text-xs text-muted-foreground">
                                        {dashboardStats.myApprovedSections} approved
                                    </p>
                                </CardContent>
                            </Card>

                            <Card>
                                <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                                    <CardTitle className="text-sm font-medium">Pending Review</CardTitle>
                                    <Clock className="h-4 w-4 text-muted-foreground" />
                                </CardHeader>
                                <CardContent>
                                    <div className="text-2xl font-bold">{dashboardStats.myPendingSections}</div>
                                    <p className="text-xs text-muted-foreground">Awaiting approval</p>
                                </CardContent>
                            </Card>

                            <Card>
                                <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                                    <CardTitle className="text-sm font-medium">My Views</CardTitle>
                                    <Eye className="h-4 w-4 text-muted-foreground" />
                                </CardHeader>
                                <CardContent>
                                    <div className="text-2xl font-bold">{dashboardStats.myTotalViews}</div>
                                    <p className="text-xs text-muted-foreground">Total views</p>
                                </CardContent>
                            </Card>

                            <Card>
                                <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                                    <CardTitle className="text-sm font-medium">My Copies</CardTitle>
                                    <Copy className="h-4 w-4 text-muted-foreground" />
                                </CardHeader>
                                <CardContent>
                                    <div className="text-2xl font-bold">{dashboardStats.myTotalCopies}</div>
                                    <p className="text-xs text-muted-foreground">Code copied</p>
                                </CardContent>
                            </Card>

                            <Card>
                                <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                                    <CardTitle className="text-sm font-medium">My Categories</CardTitle>
                                    <BarChart3 className="h-4 w-4 text-muted-foreground" />
                                </CardHeader>
                                <CardContent>
                                    <div className="text-2xl font-bold">{dashboardStats.myCategories}</div>
                                    <p className="text-xs text-muted-foreground">Categories created</p>
                                </CardContent>
                            </Card>

                            <Card>
                                <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                                    <CardTitle className="text-sm font-medium">Accessible Components</CardTitle>
                                    <Package className="h-4 w-4 text-muted-foreground" />
                                </CardHeader>
                                <CardContent>
                                    <div className="text-2xl font-bold">{dashboardStats.approvedSections}</div>
                                    <p className="text-xs text-muted-foreground">You have access to</p>
                                </CardContent>
                            </Card>
                        </>
                    )}

                    {/* For Admins - Show global stats */}
                    {userRole === 'admin' && (
                        <>
                            <Card>
                                <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                                    <CardTitle className="text-sm font-medium">Total Sections</CardTitle>
                                    <Package className="h-4 w-4 text-muted-foreground" />
                                </CardHeader>
                                <CardContent>
                                    <div className="text-2xl font-bold">{dashboardStats.totalSections}</div>
                                    <p className="text-xs text-muted-foreground">All sections</p>
                                </CardContent>
                            </Card>

                            <Card>
                                <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                                    <CardTitle className="text-sm font-medium">Approved</CardTitle>
                                    <CheckSquare className="h-4 w-4 text-muted-foreground" />
                                </CardHeader>
                                <CardContent>
                                    <div className="text-2xl font-bold">{dashboardStats.approvedSections}</div>
                                    <p className="text-xs text-muted-foreground">Published sections</p>
                                </CardContent>
                            </Card>

                            <Card>
                                <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                                    <CardTitle className="text-sm font-medium">Pending Review</CardTitle>
                                    <Clock className="h-4 w-4 text-muted-foreground" />
                                </CardHeader>
                                <CardContent>
                                    <div className="text-2xl font-bold">{dashboardStats.pendingSections}</div>
                                    <p className="text-xs text-muted-foreground">Awaiting approval</p>
                                </CardContent>
                            </Card>

                            <Card>
                                <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                                    <CardTitle className="text-sm font-medium">Rejected</CardTitle>
                                    <Package className="h-4 w-4 text-muted-foreground" />
                                </CardHeader>
                                <CardContent>
                                    <div className="text-2xl font-bold">{dashboardStats.rejectedSections}</div>
                                    <p className="text-xs text-muted-foreground">Rejected sections</p>
                                </CardContent>
                            </Card>

                            <Card>
                                <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                                    <CardTitle className="text-sm font-medium">Total Categories</CardTitle>
                                    <BarChart3 className="h-4 w-4 text-muted-foreground" />
                                </CardHeader>
                                <CardContent>
                                    <div className="text-2xl font-bold">{dashboardStats.totalCategories}</div>
                                    <p className="text-xs text-muted-foreground">All categories</p>
                                </CardContent>
                            </Card>

                            <Card>
                                <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                                    <CardTitle className="text-sm font-medium">Total Views</CardTitle>
                                    <Eye className="h-4 w-4 text-muted-foreground" />
                                </CardHeader>
                                <CardContent>
                                    <div className="text-2xl font-bold">{dashboardStats.totalViews}</div>
                                    <p className="text-xs text-muted-foreground">Component views</p>
                                </CardContent>
                            </Card>

                            <Card>
                                <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                                    <CardTitle className="text-sm font-medium">Total Copies</CardTitle>
                                    <Copy className="h-4 w-4 text-muted-foreground" />
                                </CardHeader>
                                <CardContent>
                                    <div className="text-2xl font-bold">{dashboardStats.totalCopies}</div>
                                    <p className="text-xs text-muted-foreground">Code copied</p>
                                </CardContent>
                            </Card>

                            <Card>
                                <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                                    <CardTitle className="text-sm font-medium">Total Users</CardTitle>
                                    <Users className="h-4 w-4 text-muted-foreground" />
                                </CardHeader>
                                <CardContent>
                                    <div className="text-2xl font-bold">{dashboardStats.totalUsers}</div>
                                    <p className="text-xs text-muted-foreground">
                                        {dashboardStats.totalContributors} contributors
                                    </p>
                                </CardContent>
                            </Card>
                        </>
                    )}

                    {/* For Regular Users */}
                    {userRole === 'user' && (
                        <>
                            <Card>
                                <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                                    <CardTitle className="text-sm font-medium">Available Components</CardTitle>
                                    <Package className="h-4 w-4 text-muted-foreground" />
                                </CardHeader>
                                <CardContent>
                                    <div className="text-2xl font-bold">{dashboardStats.approvedSections}</div>
                                    <p className="text-xs text-muted-foreground">You have access to</p>
                                </CardContent>
                            </Card>

                            <Card>
                                <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                                    <CardTitle className="text-sm font-medium">Total Views</CardTitle>
                                    <Eye className="h-4 w-4 text-muted-foreground" />
                                </CardHeader>
                                <CardContent>
                                    <div className="text-2xl font-bold">{dashboardStats.totalViews}</div>
                                    <p className="text-xs text-muted-foreground">Component views</p>
                                </CardContent>
                            </Card>

                            <Card>
                                <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                                    <CardTitle className="text-sm font-medium">Total Copies</CardTitle>
                                    <Copy className="h-4 w-4 text-muted-foreground" />
                                </CardHeader>
                                <CardContent>
                                    <div className="text-2xl font-bold">{dashboardStats.totalCopies}</div>
                                    <p className="text-xs text-muted-foreground">Code copied</p>
                                </CardContent>
                            </Card>
                        </>
                    )}
                </div>

                {/* Quick Actions */}
                <div className="grid gap-4 md:grid-cols-2">
                    {/* For All Users */}
                    <Card>
                        <CardHeader>
                            <CardTitle>Browse Components</CardTitle>
                            <CardDescription>
                                Explore thousands of ready-to-use UI components
                            </CardDescription>
                        </CardHeader>
                        <CardContent>
                            <Link href="/sections">
                                <Button className="w-full">
                                    <Package className="mr-2 h-4 w-4" />
                                    Browse Library
                                </Button>
                            </Link>
                        </CardContent>
                    </Card>

                    {/* For Contributors and Admins */}
                    {(userRole === 'contributor' || userRole === 'admin') && (
                        <Card>
                            <CardHeader>
                                <CardTitle>Add New Component</CardTitle>
                                <CardDescription>
                                    Create and submit a new UI component
                                </CardDescription>
                            </CardHeader>
                            <CardContent>
                                <Link href="/sections/create">
                                    <Button className="w-full">
                                        <Plus className="mr-2 h-4 w-4" />
                                        Create Section
                                    </Button>
                                </Link>
                            </CardContent>
                        </Card>
                    )}

                    {/* For Admins Only */}
                    {userRole === 'admin' && (
                        <>
                            <Card>
                                <CardHeader>
                                    <CardTitle>Review Submissions</CardTitle>
                                    <CardDescription>
                                        Approve or reject pending component submissions
                                    </CardDescription>
                                </CardHeader>
                                <CardContent>
                                    <Link href="/admin/sections?status=pending">
                                        <Button className="w-full" variant="secondary">
                                            <CheckSquare className="mr-2 h-4 w-4" />
                                            Review Pending ({dashboardStats.pendingSections})
                                        </Button>
                                    </Link>
                                </CardContent>
                            </Card>

                            <Card>
                                <CardHeader>
                                    <CardTitle>Manage Users</CardTitle>
                                    <CardDescription>
                                        View and manage user accounts and roles
                                    </CardDescription>
                                </CardHeader>
                                <CardContent>
                                    <Link href="/admin/users">
                                        <Button className="w-full" variant="secondary">
                                            <Users className="mr-2 h-4 w-4" />
                                            Manage Users
                                        </Button>
                                    </Link>
                                </CardContent>
                            </Card>
                        </>
                    )}
                </div>

                {/* Recent Activity or Popular Sections */}
                <Card>
                    <CardHeader>
                        <CardTitle>Popular Components</CardTitle>
                        <CardDescription>Most viewed components this week</CardDescription>
                    </CardHeader>
                    <CardContent>
                        <div className="flex items-center justify-center py-8 text-muted-foreground">
                            <TrendingUp className="mr-2 h-5 w-5" />
                            <span>Popular components will appear here</span>
                        </div>
                    </CardContent>
                </Card>
            </div>
        </AppLayout>
    );
}
