import { Button } from '@/components/ui/button';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Checkbox } from '@/components/ui/checkbox';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import {
    Select,
    SelectContent,
    SelectItem,
    SelectTrigger,
    SelectValue,
} from '@/components/ui/select';
import AppLayout from '@/layouts/app-layout';
import { Head, router, useForm } from '@inertiajs/react';
import { ArrowLeft } from 'lucide-react';
import { FormEvent } from 'react';

export default function CreateUser() {
    const { data, setData, processing, errors } = useForm({
        name: '',
        email: '',
        password: '',
        password_confirmation: '',
        role: 'user',
        access_web: false,
        access_dashboard: false,
    });

    const handleSubmit = (e: FormEvent) => {
        e.preventDefault();

        // Convert checkbox values to access_type enum
        let access_type = 'both';
        if (data.role === 'contributor') {
            if (data.access_web && !data.access_dashboard) {
                access_type = 'web';
            } else if (!data.access_web && data.access_dashboard) {
                access_type = 'dashboard';
            } else if (data.access_web && data.access_dashboard) {
                access_type = 'both';
            }
        }

        router.post('/admin/users', {
            ...data,
            access_type,
        });
    };

    return (
        <AppLayout>
            <Head title="Create User" />

            <div className="container mx-auto px-4 py-8 max-w-2xl">
                {/* Header */}
                <div className="mb-8">
                    <Button
                        variant="ghost"
                        className="mb-4"
                        onClick={() => window.history.back()}
                    >
                        <ArrowLeft className="mr-2 h-4 w-4" />
                        Back
                    </Button>
                    <h1 className="text-4xl font-bold mb-2">Create New User</h1>
                    <p className="text-muted-foreground">
                        Add a new user to the system
                    </p>
                </div>

                <form onSubmit={handleSubmit}>
                    <Card>
                        <CardHeader>
                            <CardTitle>User Information</CardTitle>
                            <CardDescription>
                                Fill in the details for the new user
                            </CardDescription>
                        </CardHeader>
                        <CardContent className="space-y-4">
                            {/* Name */}
                            <div className="space-y-2">
                                <Label htmlFor="name">
                                    Name <span className="text-red-500">*</span>
                                </Label>
                                <Input
                                    id="name"
                                    value={data.name}
                                    onChange={(e) => setData('name', e.target.value)}
                                    placeholder="John Doe"
                                    required
                                />
                                {errors.name && (
                                    <p className="text-sm text-red-500">{errors.name}</p>
                                )}
                            </div>

                            {/* Email */}
                            <div className="space-y-2">
                                <Label htmlFor="email">
                                    Email <span className="text-red-500">*</span>
                                </Label>
                                <Input
                                    id="email"
                                    type="email"
                                    value={data.email}
                                    onChange={(e) => setData('email', e.target.value)}
                                    placeholder="john@example.com"
                                    required
                                />
                                {errors.email && (
                                    <p className="text-sm text-red-500">{errors.email}</p>
                                )}
                            </div>

                            {/* Password */}
                            <div className="space-y-2">
                                <Label htmlFor="password">
                                    Password <span className="text-red-500">*</span>
                                </Label>
                                <Input
                                    id="password"
                                    type="password"
                                    value={data.password}
                                    onChange={(e) => setData('password', e.target.value)}
                                    placeholder="••••••••"
                                    required
                                />
                                {errors.password && (
                                    <p className="text-sm text-red-500">{errors.password}</p>
                                )}
                            </div>

                            {/* Confirm Password */}
                            <div className="space-y-2">
                                <Label htmlFor="password_confirmation">
                                    Confirm Password <span className="text-red-500">*</span>
                                </Label>
                                <Input
                                    id="password_confirmation"
                                    type="password"
                                    value={data.password_confirmation}
                                    onChange={(e) => setData('password_confirmation', e.target.value)}
                                    placeholder="••••••••"
                                    required
                                />
                            </div>

                            {/* Role */}
                            <div className="space-y-2">
                                <Label htmlFor="role">
                                    Role <span className="text-red-500">*</span>
                                </Label>
                                <Select
                                    value={data.role}
                                    onValueChange={(value) => setData('role', value)}
                                    required
                                >
                                    <SelectTrigger>
                                        <SelectValue />
                                    </SelectTrigger>
                                    <SelectContent>
                                        <SelectItem value="user">User</SelectItem>
                                        <SelectItem value="contributor">Contributor</SelectItem>
                                        <SelectItem value="admin">Admin</SelectItem>
                                    </SelectContent>
                                </Select>
                                {errors.role && (
                                    <p className="text-sm text-red-500">{errors.role}</p>
                                )}
                                <p className="text-sm text-muted-foreground">
                                    <strong>User:</strong> Can browse sections<br />
                                    <strong>Contributor:</strong> Can create and manage their own sections<br />
                                    <strong>Admin:</strong> Full access to all features
                                </p>
                            </div>

                            {/* Access Type - Only for Contributors */}
                            {data.role === 'contributor' && (
                                <div className="space-y-2">
                                    <Label>
                                        Access Type <span className="text-red-500">*</span>
                                    </Label>
                                    <div className="space-y-3">
                                        <div className="flex items-center space-x-2">
                                            <Checkbox
                                                id="access_web"
                                                checked={data.access_web}
                                                onCheckedChange={(checked) => setData('access_web', checked as boolean)}
                                            />
                                            <Label htmlFor="access_web" className="font-normal cursor-pointer">
                                                Web - Can access web category sections
                                            </Label>
                                        </div>
                                        <div className="flex items-center space-x-2">
                                            <Checkbox
                                                id="access_dashboard"
                                                checked={data.access_dashboard}
                                                onCheckedChange={(checked) => setData('access_dashboard', checked as boolean)}
                                            />
                                            <Label htmlFor="access_dashboard" className="font-normal cursor-pointer">
                                                Dashboard - Can access dashboard category sections
                                            </Label>
                                        </div>
                                    </div>
                                    {'access_type' in errors && errors.access_type && (
                                        // @ts-expect-error - access_type is added dynamically from backend validation
                                        <p className="text-sm text-red-500">{String(errors.access_type)}</p>
                                    )}
                                    {!data.access_web && !data.access_dashboard && (
                                        <p className="text-sm text-amber-600">
                                            Please select at least one access type
                                        </p>
                                    )}
                                </div>
                            )}
                        </CardContent>
                    </Card>

                    {/* Submit */}
                    <div className="flex gap-2 justify-end mt-6">
                        <Button
                            type="button"
                            variant="outline"
                            onClick={() => window.history.back()}
                        >
                            Cancel
                        </Button>
                        <Button type="submit" disabled={processing}>
                            {processing ? 'Creating...' : 'Create User'}
                        </Button>
                    </div>
                </form>
            </div>
        </AppLayout>
    );
}
