import { Button } from '@/components/ui/button';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { RadioGroup, RadioGroupItem } from '@/components/ui/radio-group';
import { Textarea } from '@/components/ui/textarea';
import AppLayout from '@/layouts/app-layout';
import { Head, useForm } from '@inertiajs/react';
import { ArrowLeft } from 'lucide-react';
import { FormEvent } from 'react';

interface Props {
    accessibleTypes: string[];
}

export default function CreateCategory({ accessibleTypes }: Props) {
    const { data, setData, post, processing, errors } = useForm({
        name: '',
        type: accessibleTypes[0] || 'web',
        description: '',
        icon: '',
        color: '#3b82f6',
    });

    const handleSubmit = (e: FormEvent) => {
        e.preventDefault();
        post('/admin/categories');
    };

    return (
        <AppLayout>
            <Head title="Create Category" />

            <div className="container mx-auto px-4 py-8 max-w-2xl">
                {/* Header */}
                <div className="mb-8">
                    <Button
                        variant="ghost"
                        className="mb-4"
                        onClick={() => window.history.back()}
                    >
                        <ArrowLeft className="mr-2 h-4 w-4" />
                        Back to Categories
                    </Button>
                    <h1 className="text-4xl font-bold mb-2">Create Category</h1>
                    <p className="text-muted-foreground">
                        Add a new category to organize your components
                    </p>
                </div>

                {/* Form */}
                <Card>
                    <CardHeader>
                        <CardTitle>Category Details</CardTitle>
                        <CardDescription>
                            Fill in the information below to create a new category
                        </CardDescription>
                    </CardHeader>
                    <CardContent>
                        <form onSubmit={handleSubmit} className="space-y-6">
                            {/* Name */}
                            <div className="space-y-2">
                                <Label htmlFor="name">
                                    Name <span className="text-red-500">*</span>
                                </Label>
                                <Input
                                    id="name"
                                    value={data.name}
                                    onChange={(e) => setData('name', e.target.value)}
                                    placeholder="e.g., Hero, Features, Testimonials"
                                    required
                                />
                                {errors.name && (
                                    <p className="text-sm text-red-500">{errors.name}</p>
                                )}
                            </div>


                            {/* Type - Only show if user has access to both types */}
                            {accessibleTypes.length > 1 && (
                                <div className="space-y-2">
                                    <Label htmlFor="type">
                                        Type <span className="text-red-500">*</span>
                                    </Label>
                                    <RadioGroup
                                        value={data.type}
                                        onValueChange={(value) => setData('type', value)}
                                        className="flex gap-4"
                                    >
                                        {accessibleTypes.includes('web') && (
                                            <div className="flex items-center space-x-2">
                                                <RadioGroupItem value="web" id="type-web" />
                                                <Label htmlFor="type-web" className="font-normal cursor-pointer">
                                                    Web
                                                </Label>
                                            </div>
                                        )}
                                        {accessibleTypes.includes('dashboard') && (
                                            <div className="flex items-center space-x-2">
                                                <RadioGroupItem value="dashboard" id="type-dashboard" />
                                                <Label htmlFor="type-dashboard" className="font-normal cursor-pointer">
                                                    Dashboard
                                                </Label>
                                            </div>
                                        )}
                                    </RadioGroup>
                                    {errors.type && (
                                        <p className="text-sm text-red-500">{errors.type}</p>
                                    )}
                                    <p className="text-xs text-muted-foreground">
                                        Select whether this category is for web sections or dashboard sections
                                    </p>
                                </div>
                            )}

                            {/* Description */}
                            <div className="space-y-2">
                                <Label htmlFor="description">Description</Label>
                                <Textarea
                                    id="description"
                                    value={data.description}
                                    onChange={(e) => setData('description', e.target.value)}
                                    placeholder="Brief description of this category"
                                    rows={3}
                                />
                                {errors.description && (
                                    <p className="text-sm text-red-500">{errors.description}</p>
                                )}
                            </div>

                            {/* Icon */}
                            <div className="space-y-2">
                                <Label htmlFor="icon">Icon (Lucide Icon Name)</Label>
                                <Input
                                    id="icon"
                                    value={data.icon}
                                    onChange={(e) => setData('icon', e.target.value)}
                                    placeholder="e.g., Rocket, Users, Mail"
                                />
                                {errors.icon && (
                                    <p className="text-sm text-red-500">{errors.icon}</p>
                                )}
                                <p className="text-xs text-muted-foreground">
                                    Visit{' '}
                                    <a
                                        href="https://lucide.dev/icons/"
                                        target="_blank"
                                        rel="noopener noreferrer"
                                        className="text-primary hover:underline"
                                    >
                                        lucide.dev
                                    </a>{' '}
                                    for available icons
                                </p>
                            </div>

                            {/* Color */}
                            <div className="space-y-2">
                                <Label htmlFor="color">Color</Label>
                                <div className="flex gap-2 items-center">
                                    <Input
                                        id="color"
                                        type="color"
                                        value={data.color}
                                        onChange={(e) => setData('color', e.target.value)}
                                        className="w-20 h-10"
                                    />
                                    <Input
                                        type="text"
                                        value={data.color}
                                        onChange={(e) => setData('color', e.target.value)}
                                        placeholder="#3b82f6"
                                        className="flex-1"
                                    />
                                </div>
                                {errors.color && (
                                    <p className="text-sm text-red-500">{errors.color}</p>
                                )}
                            </div>

                            {/* Submit */}
                            <div className="flex gap-2 justify-end">
                                <Button
                                    type="button"
                                    variant="outline"
                                    onClick={() => window.history.back()}
                                >
                                    Cancel
                                </Button>
                                <Button type="submit" disabled={processing}>
                                    {processing ? 'Creating...' : 'Create Category'}
                                </Button>
                            </div>
                        </form>
                    </CardContent>
                </Card>
            </div>
        </AppLayout>
    );
}
