import { useEffect, useState } from 'react';
import { codeToHtml } from 'shiki';

interface CodeBlockProps {
    code: string;
    language: string;
    theme?: 'github-dark' | 'github-light' | 'nord' | 'one-dark-pro' | 'dracula' | 'tokyo-night';
    showLineNumbers?: boolean;
}

export function CodeBlock({
    code,
    language,
    theme = 'github-dark',
    showLineNumbers = true
}: CodeBlockProps) {
    const [html, setHtml] = useState<string>('');
    const [isLoading, setIsLoading] = useState(true);

    useEffect(() => {
        const highlightCode = async () => {
            setIsLoading(true);
            try {
                const highlighted = await codeToHtml(code, {
                    lang: language,
                    theme: theme,
                    transformers: showLineNumbers ? [
                        {
                            line(node, line) {
                                node.properties['data-line'] = line;
                                if (line === 1) {
                                    node.properties.style = 'counter-reset: line 0;';
                                }
                            },
                        },
                    ] : [],
                });
                setHtml(highlighted);
            } catch (error) {
                console.error('Failed to highlight code:', error);
                // Fallback to plain text
                setHtml(`<pre><code>${code}</code></pre>`);
            } finally {
                setIsLoading(false);
            }
        };

        highlightCode();
    }, [code, language, theme, showLineNumbers]);

    if (isLoading) {
        return (
            <div className="bg-muted p-4 rounded-lg overflow-x-auto max-h-[600px] animate-pulse">
                <div className="h-4 bg-muted-foreground/20 rounded w-3/4 mb-2"></div>
                <div className="h-4 bg-muted-foreground/20 rounded w-1/2 mb-2"></div>
                <div className="h-4 bg-muted-foreground/20 rounded w-5/6 mb-2"></div>
                <div className="h-4 bg-muted-foreground/20 rounded w-2/3"></div>
            </div>
        );
    }

    return (
        <div
            className="shiki-wrapper overflow-x-auto max-h-[600px] rounded-lg"
            style={{
                ...(showLineNumbers && {
                    counterReset: 'line',
                }),
            }}
        >
            <style dangerouslySetInnerHTML={{
                __html: `
                    .shiki-wrapper pre {
                        margin: 0;
                        padding: 1rem;
                        overflow-x: auto;
                    }
                    
                    .shiki-wrapper code {
                        font-family: 'Fira Code', 'Cascadia Code', 'JetBrains Mono', 'Consolas', monospace;
                        font-size: 0.875rem;
                        line-height: 1;
                        counter-reset: line;
                    }
                    
                    ${showLineNumbers ? `
                        .shiki-wrapper code .line::before {
                            counter-increment: line;
                            content: counter(line);
                            display: inline-block;
                            width: 1.5rem;
                            margin-right: 1.5rem;
                            text-align: right;
                            color: rgba(115, 138, 148, 0.4);
                            user-select: none;
                        }
                    ` : ''}
                    
                    .shiki-wrapper code .line {
                        display: block;
                        line-height: 1;
                        margin: 0;
                        padding: 0;
                    }
                    
                    .shiki-wrapper::-webkit-scrollbar {
                        height: 8px;
                    }
                    
                    .shiki-wrapper::-webkit-scrollbar-track {
                        background: rgba(0, 0, 0, 0.1);
                        border-radius: 4px;
                    }
                    
                    .shiki-wrapper::-webkit-scrollbar-thumb {
                        background: rgba(0, 0, 0, 0.3);
                        border-radius: 4px;
                    }
                    
                    .shiki-wrapper::-webkit-scrollbar-thumb:hover {
                        background: rgba(0, 0, 0, 0.5);
                    }
                `
            }} />
            <div dangerouslySetInnerHTML={{ __html: html }} />
        </div>
    );
}
