<?php

namespace Database\Seeders;

use App\Models\Category;
use App\Models\Section;
use App\Models\User;
use Illuminate\Database\Seeder;

class SectionSeeder extends Seeder
{
    /**
     * Run the database seeds.
     */
    public function run(): void
    {
        // Get existing users (created by UserSeeder)
        $admin = User::where('email', 'ikholil.bd@gmail.com')->first();
        $contributor = User::where('email', 'btouha007@gmail.com')->first();

        if (!$admin || !$contributor) {
            $this->command->error('Users not found. Please run UserSeeder first.');

            return;
        }

        // Check if sections already exist
        if (Section::count() > 0) {
            $this->command->info('Sections already exist. Skipping section seeding.');

            return;
        }

        // Get categories
        $heroCategory = Category::where('slug', 'hero')->first();
        $teamCategory = Category::where('slug', 'team')->first();
        $contactCategory = Category::where('slug', 'contact')->first();
        $featuresCategory = Category::where('slug', 'features')->first();
        $aboutCategory = Category::where('slug', 'about')->first();
        $blogsCategory = Category::where('slug', 'blogs')->first();
        $testimonialsCategory = Category::where('slug', 'testimonials')->first();
        $pricingCategory = Category::where('slug', 'pricing')->first();
        $faqCategory = Category::where('slug', 'faq')->first();
        $footerCategory = Category::where('slug', 'footer')->first();
        $navbarCategory = Category::where('slug', 'navbar')->first();
        $ctaCategory = Category::where('slug', 'cta')->first();

        // Sample sections data
        $sectionsData = [
            [
                'name' => 'Modern Hero Section',
                'description' => 'A beautiful hero section with gradient background and call-to-action buttons',
                'category_id' => $heroCategory->id,
                'tags' => ['gradient', 'cta', 'modern'],
                'user_id' => $admin->id,
                'status' => 'approved',
                'approved_by' => $admin->id,
                'approved_at' => now(),
                'versions' => [
                    [
                        'framework' => 'html',
                        'styling' => 'tailwind',
                        'code' => '<section class="bg-gradient-to-r from-purple-500 to-pink-500 py-20">
    <div class="container mx-auto px-4 text-center text-white">
        <h1 class="text-5xl font-bold mb-4">Welcome to Our Platform</h1>
        <p class="text-xl mb-8">Build amazing things with our components</p>
        <div class="flex gap-4 justify-center">
            <button class="bg-white text-purple-600 px-8 py-3 rounded-lg font-semibold hover:bg-gray-100">Get Started</button>
            <button class="border-2 border-white px-8 py-3 rounded-lg font-semibold hover:bg-white hover:text-purple-600">Learn More</button>
        </div>
    </div>
</section>',
                        'dependencies' => ['tailwindcss'],
                    ],
                    [
                        'framework' => 'react',
                        'styling' => 'tailwind',
                        'code' => 'export default function HeroSection() {
    return (
        <section className="bg-gradient-to-r from-purple-500 to-pink-500 py-20">
            <div className="container mx-auto px-4 text-center text-white">
                <h1 className="text-5xl font-bold mb-4">Welcome to Our Platform</h1>
                <p className="text-xl mb-8">Build amazing things with our components</p>
                <div className="flex gap-4 justify-center">
                    <button className="bg-white text-purple-600 px-8 py-3 rounded-lg font-semibold hover:bg-gray-100">
                        Get Started
                    </button>
                    <button className="border-2 border-white px-8 py-3 rounded-lg font-semibold hover:bg-white hover:text-purple-600">
                        Learn More
                    </button>
                </div>
            </div>
        </section>
    );
}',
                        'dependencies' => ['react', 'tailwindcss'],
                    ],
                ],
            ],
            [
                'name' => 'Team Grid Section',
                'description' => 'Display your team members in a responsive grid layout',
                'category_id' => $teamCategory->id,
                'tags' => ['grid', 'cards', 'responsive'],
                'user_id' => $contributor->id,
                'status' => 'approved',
                'approved_by' => $admin->id,
                'approved_at' => now(),
                'versions' => [
                    [
                        'framework' => 'html',
                        'styling' => 'tailwind',
                        'code' => '<section class="py-16 bg-gray-50">
    <div class="container mx-auto px-4">
        <h2 class="text-4xl font-bold text-center mb-12">Our Team</h2>
        <div class="grid grid-cols-1 md:grid-cols-3 gap-8">
            <div class="bg-white rounded-lg shadow-lg p-6 text-center">
                <img src="https://via.placeholder.com/150" alt="Team member" class="w-32 h-32 rounded-full mx-auto mb-4">
                <h3 class="text-xl font-bold mb-2">John Doe</h3>
                <p class="text-gray-600">CEO & Founder</p>
            </div>
        </div>
    </div>
</section>',
                        'dependencies' => ['tailwindcss'],
                    ],
                    [
                        'framework' => 'react',
                        'styling' => 'tailwind',
                        'code' => 'export default function TeamSection() {
    const team = [
        { name: "John Doe", role: "CEO & Founder", image: "https://via.placeholder.com/150" },
    ];

    return (
        <section className="py-16 bg-gray-50">
            <div className="container mx-auto px-4">
                <h2 className="text-4xl font-bold text-center mb-12">Our Team</h2>
                <div className="grid grid-cols-1 md:grid-cols-3 gap-8">
                    {team.map((member, index) => (
                        <div key={index} className="bg-white rounded-lg shadow-lg p-6 text-center">
                            <img src={member.image} alt={member.name} className="w-32 h-32 rounded-full mx-auto mb-4" />
                            <h3 className="text-xl font-bold mb-2">{member.name}</h3>
                            <p className="text-gray-600">{member.role}</p>
                        </div>
                    ))}
                </div>
            </div>
        </section>
    );
}',
                        'dependencies' => ['react', 'tailwindcss'],
                    ],
                ],
            ],
            [
                'name' => 'Contact Form',
                'description' => 'Simple and elegant contact form with validation',
                'category_id' => $contactCategory->id,
                'tags' => ['form', 'validation', 'input'],
                'user_id' => $contributor->id,
                'status' => 'pending',
                'versions' => [
                    [
                        'framework' => 'html',
                        'styling' => 'tailwind',
                        'code' => '<section class="py-16">
    <div class="container mx-auto px-4 max-w-2xl">
        <h2 class="text-4xl font-bold text-center mb-12">Contact Us</h2>
        <form class="space-y-6">
            <div>
                <label class="block text-sm font-medium mb-2">Name</label>
                <input type="text" class="w-full px-4 py-2 border rounded-lg focus:ring-2 focus:ring-blue-500" required>
            </div>
            <div>
                <label class="block text-sm font-medium mb-2">Email</label>
                <input type="email" class="w-full px-4 py-2 border rounded-lg focus:ring-2 focus:ring-blue-500" required>
            </div>
            <div>
                <label class="block text-sm font-medium mb-2">Message</label>
                <textarea rows="4" class="w-full px-4 py-2 border rounded-lg focus:ring-2 focus:ring-blue-500" required></textarea>
            </div>
            <button type="submit" class="w-full bg-blue-600 text-white py-3 rounded-lg font-semibold hover:bg-blue-700">Send Message</button>
        </form>
    </div>
</section>',
                        'dependencies' => ['tailwindcss'],
                    ],
                ],
            ],
            [
                'name' => 'Feature Cards',
                'description' => 'Showcase your product features with icon cards',
                'category_id' => $featuresCategory->id,
                'tags' => ['cards', 'icons', 'grid'],
                'user_id' => $admin->id,
                'status' => 'approved',
                'approved_by' => $admin->id,
                'approved_at' => now(),
                'versions' => [
                    [
                        'framework' => 'html',
                        'styling' => 'tailwind',
                        'code' => '<section class="py-16">
    <div class="container mx-auto px-4">
        <h2 class="text-4xl font-bold text-center mb-12">Features</h2>
        <div class="grid grid-cols-1 md:grid-cols-3 gap-8">
            <div class="p-6 border rounded-lg hover:shadow-lg transition">
                <div class="w-12 h-12 bg-blue-100 rounded-lg flex items-center justify-center mb-4">
                    <svg class="w-6 h-6 text-blue-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M13 10V3L4 14h7v7l9-11h-7z"></path>
                    </svg>
                </div>
                <h3 class="text-xl font-bold mb-2">Fast Performance</h3>
                <p class="text-gray-600">Lightning fast load times for better user experience</p>
            </div>
        </div>
    </div>
</section>',
                        'dependencies' => ['tailwindcss'],
                    ],
                    [
                        'framework' => 'react',
                        'styling' => 'tailwind',
                        'code' => 'export default function FeaturesSection() {
    const features = [
        {
            title: "Fast Performance",
            description: "Lightning fast load times for better user experience",
            icon: "⚡"
        }
    ];

    return (
        <section className="py-16">
            <div className="container mx-auto px-4">
                <h2 className="text-4xl font-bold text-center mb-12">Features</h2>
                <div className="grid grid-cols-1 md:grid-cols-3 gap-8">
                    {features.map((feature, index) => (
                        <div key={index} className="p-6 border rounded-lg hover:shadow-lg transition">
                            <div className="w-12 h-12 bg-blue-100 rounded-lg flex items-center justify-center mb-4 text-2xl">
                                {feature.icon}
                            </div>
                            <h3 className="text-xl font-bold mb-2">{feature.title}</h3>
                            <p className="text-gray-600">{feature.description}</p>
                        </div>
                    ))}
                </div>
            </div>
        </section>
    );
}',
                        'dependencies' => ['react', 'tailwindcss'],
                    ],
                ],
            ],
            // New sections start here
            [
                'name' => 'Animated Hero with Video',
                'description' => 'Eye-catching hero section with background video and animated elements',
                'category_id' => $heroCategory->id,
                'tags' => ['video', 'animation', 'modern'],
                'user_id' => $contributor->id,
                'status' => 'approved',
                'approved_by' => $admin->id,
                'approved_at' => now(),
                'versions' => [
                    [
                        'framework' => 'html',
                        'styling' => 'tailwind',
                        'code' => '<section class="relative h-screen flex items-center justify-center overflow-hidden">
    <video autoplay muted loop class="absolute inset-0 w-full h-full object-cover">
        <source src="hero-video.mp4" type="video/mp4">
    </video>
    <div class="relative z-10 text-center text-white px-4">
        <h1 class="text-6xl font-bold mb-6 animate-fade-in">Transform Your Business</h1>
        <p class="text-2xl mb-8 animate-slide-up">Innovative solutions for modern challenges</p>
        <button class="bg-blue-600 hover:bg-blue-700 px-10 py-4 rounded-full text-lg font-semibold transition transform hover:scale-105">Start Free Trial</button>
    </div>
    <div class="absolute inset-0 bg-black opacity-40"></div>
</section>',
                        'dependencies' => ['tailwindcss'],
                    ],
                ],
            ],
            [
                'name' => 'About Us Timeline',
                'description' => 'Company history displayed in an elegant timeline format',
                'category_id' => $aboutCategory->id,
                'tags' => ['timeline', 'history', 'story'],
                'user_id' => $admin->id,
                'status' => 'approved',
                'approved_by' => $admin->id,
                'approved_at' => now(),
                'versions' => [
                    [
                        'framework' => 'html',
                        'styling' => 'tailwind',
                        'code' => '<section class="py-20 bg-gradient-to-b from-gray-50 to-white">
    <div class="container mx-auto px-4">
        <h2 class="text-5xl font-bold text-center mb-16">Our Journey</h2>
        <div class="max-w-4xl mx-auto">
            <div class="relative border-l-4 border-blue-500 pl-8 ml-4 space-y-12">
                <div class="relative">
                    <div class="absolute -left-12 w-8 h-8 bg-blue-500 rounded-full border-4 border-white"></div>
                    <div class="bg-white p-6 rounded-lg shadow-lg">
                        <span class="text-blue-600 font-bold">2020</span>
                        <h3 class="text-2xl font-bold mt-2 mb-3">Company Founded</h3>
                        <p class="text-gray-600">Started with a vision to revolutionize the industry</p>
                    </div>
                </div>
                <div class="relative">
                    <div class="absolute -left-12 w-8 h-8 bg-blue-500 rounded-full border-4 border-white"></div>
                    <div class="bg-white p-6 rounded-lg shadow-lg">
                        <span class="text-blue-600 font-bold">2022</span>
                        <h3 class="text-2xl font-bold mt-2 mb-3">Major Milestone</h3>
                        <p class="text-gray-600">Reached 10,000 satisfied customers worldwide</p>
                    </div>
                </div>
            </div>
        </div>
    </div>
</section>',
                        'dependencies' => ['tailwindcss'],
                    ],
                ],
            ],
            [
                'name' => 'Modern Blog Grid',
                'description' => 'Responsive blog post grid with hover effects and categories',
                'category_id' => $blogsCategory->id,
                'tags' => ['grid', 'cards', 'hover'],
                'user_id' => $contributor->id,
                'status' => 'approved',
                'approved_by' => $admin->id,
                'approved_at' => now(),
                'versions' => [
                    [
                        'framework' => 'react',
                        'styling' => 'tailwind',
                        'code' => 'export default function BlogGrid() {
    const posts = [
        { title: "Getting Started with React", category: "Development", image: "blog1.jpg", date: "Dec 20, 2024" },
        { title: "Design Trends 2024", category: "Design", image: "blog2.jpg", date: "Dec 18, 2024" },
        { title: "SEO Best Practices", category: "Marketing", image: "blog3.jpg", date: "Dec 15, 2024" },
    ];

    return (
        <section className="py-20 bg-gray-50">
            <div className="container mx-auto px-4">
                <h2 className="text-5xl font-bold text-center mb-16">Latest Articles</h2>
                <div className="grid grid-cols-1 md:grid-cols-3 gap-8">
                    {posts.map((post, index) => (
                        <article key={index} className="bg-white rounded-xl overflow-hidden shadow-lg hover:shadow-2xl transition-all duration-300 transform hover:-translate-y-2">
                            <img src={post.image} alt={post.title} className="w-full h-48 object-cover" />
                            <div className="p-6">
                                <span className="text-blue-600 text-sm font-semibold">{post.category}</span>
                                <h3 className="text-2xl font-bold mt-2 mb-3">{post.title}</h3>
                                <p className="text-gray-500 text-sm">{post.date}</p>
                            </div>
                        </article>
                    ))}
                </div>
            </div>
        </section>
    );
}',
                        'dependencies' => ['react', 'tailwindcss'],
                    ],
                ],
            ],
            [
                'name' => 'Testimonial Carousel',
                'description' => 'Beautiful testimonial slider with customer reviews',
                'category_id' => $testimonialsCategory->id,
                'tags' => ['carousel', 'reviews', 'slider'],
                'user_id' => $admin->id,
                'status' => 'approved',
                'approved_by' => $admin->id,
                'approved_at' => now(),
                'versions' => [
                    [
                        'framework' => 'html',
                        'styling' => 'tailwind',
                        'code' => '<section class="py-20 bg-gradient-to-r from-purple-600 to-blue-600">
    <div class="container mx-auto px-4">
        <h2 class="text-5xl font-bold text-center text-white mb-16">What Our Clients Say</h2>
        <div class="max-w-4xl mx-auto bg-white rounded-2xl shadow-2xl p-12">
            <div class="text-center">
                <img src="avatar.jpg" alt="Client" class="w-24 h-24 rounded-full mx-auto mb-6 border-4 border-purple-200">
                <p class="text-2xl text-gray-700 italic mb-6">"This product has completely transformed how we do business. Highly recommended!"</p>
                <h4 class="text-xl font-bold text-gray-900">Sarah Johnson</h4>
                <p class="text-gray-600">CEO, Tech Innovations</p>
                <div class="flex justify-center mt-4 text-yellow-400">
                    <span class="text-2xl">★★★★★</span>
                </div>
            </div>
        </div>
    </div>
</section>',
                        'dependencies' => ['tailwindcss'],
                    ],
                ],
            ],
            [
                'name' => 'Pricing Table Pro',
                'description' => 'Professional pricing table with feature comparison',
                'category_id' => $pricingCategory->id,
                'tags' => ['pricing', 'comparison', 'plans'],
                'user_id' => $contributor->id,
                'status' => 'approved',
                'approved_by' => $admin->id,
                'approved_at' => now(),
                'versions' => [
                    [
                        'framework' => 'react',
                        'styling' => 'tailwind',
                        'code' => 'export default function PricingTable() {
    const plans = [
        { name: "Starter", price: "$9", features: ["5 Projects", "10GB Storage", "Email Support"], popular: false },
        { name: "Pro", price: "$29", features: ["Unlimited Projects", "100GB Storage", "Priority Support", "Advanced Analytics"], popular: true },
        { name: "Enterprise", price: "$99", features: ["Unlimited Everything", "Dedicated Support", "Custom Integration", "SLA"], popular: false },
    ];

    return (
        <section className="py-20 bg-gray-50">
            <div className="container mx-auto px-4">
                <h2 className="text-5xl font-bold text-center mb-4">Choose Your Plan</h2>
                <p className="text-xl text-gray-600 text-center mb-16">Select the perfect plan for your needs</p>
                <div className="grid grid-cols-1 md:grid-cols-3 gap-8 max-w-6xl mx-auto">
                    {plans.map((plan, index) => (
                        <div key={index} className={`bg-white rounded-2xl shadow-xl p-8 ${plan.popular ? "ring-4 ring-blue-500 transform scale-105" : ""}`}>
                            {plan.popular && <span className="bg-blue-500 text-white px-4 py-1 rounded-full text-sm font-semibold">Most Popular</span>}
                            <h3 className="text-3xl font-bold mt-4">{plan.name}</h3>
                            <div className="text-5xl font-bold my-6">{plan.price}<span className="text-xl text-gray-500">/mo</span></div>
                            <ul className="space-y-4 mb-8">
                                {plan.features.map((feature, i) => (
                                    <li key={i} className="flex items-center"><span className="text-green-500 mr-2">✓</span>{feature}</li>
                                ))}
                            </ul>
                            <button className={`w-full py-3 rounded-lg font-semibold ${plan.popular ? "bg-blue-600 text-white hover:bg-blue-700" : "bg-gray-100 hover:bg-gray-200"}`}>Get Started</button>
                        </div>
                    ))}
                </div>
            </div>
        </section>
    );
}',
                        'dependencies' => ['react', 'tailwindcss'],
                    ],
                ],
            ],
            [
                'name' => 'FAQ Accordion',
                'description' => 'Interactive FAQ section with smooth accordion animation',
                'category_id' => $faqCategory->id,
                'tags' => ['accordion', 'faq', 'interactive'],
                'user_id' => $admin->id,
                'status' => 'approved',
                'approved_by' => $admin->id,
                'approved_at' => now(),
                'versions' => [
                    [
                        'framework' => 'html',
                        'styling' => 'tailwind',
                        'code' => '<section class="py-20">
    <div class="container mx-auto px-4 max-w-4xl">
        <h2 class="text-5xl font-bold text-center mb-16">Frequently Asked Questions</h2>
        <div class="space-y-4">
            <details class="bg-white rounded-lg shadow-md group">
                <summary class="p-6 cursor-pointer font-semibold text-lg flex justify-between items-center">
                    How does the free trial work?
                    <span class="transform group-open:rotate-180 transition-transform">▼</span>
                </summary>
                <div class="px-6 pb-6 text-gray-600">
                    You get full access to all features for 14 days. No credit card required.
                </div>
            </details>
            <details class="bg-white rounded-lg shadow-md group">
                <summary class="p-6 cursor-pointer font-semibold text-lg flex justify-between items-center">
                    Can I cancel anytime?
                    <span class="transform group-open:rotate-180 transition-transform">▼</span>
                </summary>
                <div class="px-6 pb-6 text-gray-600">
                    Yes, you can cancel your subscription at any time with no penalties.
                </div>
            </details>
            <details class="bg-white rounded-lg shadow-md group">
                <summary class="p-6 cursor-pointer font-semibold text-lg flex justify-between items-center">
                    What payment methods do you accept?
                    <span class="transform group-open:rotate-180 transition-transform">▼</span>
                </summary>
                <div class="px-6 pb-6 text-gray-600">
                    We accept all major credit cards, PayPal, and bank transfers.
                </div>
            </details>
        </div>
    </div>
</section>',
                        'dependencies' => ['tailwindcss'],
                    ],
                ],
            ],
            [
                'name' => 'Modern Footer',
                'description' => 'Comprehensive footer with links, social media, and newsletter',
                'category_id' => $footerCategory->id,
                'tags' => ['footer', 'links', 'social'],
                'user_id' => $contributor->id,
                'status' => 'approved',
                'approved_by' => $admin->id,
                'approved_at' => now(),
                'versions' => [
                    [
                        'framework' => 'html',
                        'styling' => 'tailwind',
                        'code' => '<footer class="bg-gray-900 text-white py-16">
    <div class="container mx-auto px-4">
        <div class="grid grid-cols-1 md:grid-cols-4 gap-8 mb-8">
            <div>
                <h3 class="text-2xl font-bold mb-4">Company</h3>
                <ul class="space-y-2">
                    <li><a href="#" class="text-gray-400 hover:text-white transition">About Us</a></li>
                    <li><a href="#" class="text-gray-400 hover:text-white transition">Careers</a></li>
                    <li><a href="#" class="text-gray-400 hover:text-white transition">Press</a></li>
                </ul>
            </div>
            <div>
                <h3 class="text-2xl font-bold mb-4">Product</h3>
                <ul class="space-y-2">
                    <li><a href="#" class="text-gray-400 hover:text-white transition">Features</a></li>
                    <li><a href="#" class="text-gray-400 hover:text-white transition">Pricing</a></li>
                    <li><a href="#" class="text-gray-400 hover:text-white transition">Security</a></li>
                </ul>
            </div>
            <div>
                <h3 class="text-2xl font-bold mb-4">Resources</h3>
                <ul class="space-y-2">
                    <li><a href="#" class="text-gray-400 hover:text-white transition">Blog</a></li>
                    <li><a href="#" class="text-gray-400 hover:text-white transition">Documentation</a></li>
                    <li><a href="#" class="text-gray-400 hover:text-white transition">Support</a></li>
                </ul>
            </div>
            <div>
                <h3 class="text-2xl font-bold mb-4">Newsletter</h3>
                <p class="text-gray-400 mb-4">Stay updated with our latest news</p>
                <input type="email" placeholder="Your email" class="w-full px-4 py-2 rounded-lg bg-gray-800 border border-gray-700 focus:border-blue-500 focus:outline-none mb-2">
                <button class="w-full bg-blue-600 hover:bg-blue-700 px-4 py-2 rounded-lg font-semibold transition">Subscribe</button>
            </div>
        </div>
        <div class="border-t border-gray-800 pt-8 flex flex-col md:flex-row justify-between items-center">
            <p class="text-gray-400">© 2024 Your Company. All rights reserved.</p>
            <div class="flex gap-4 mt-4 md:mt-0">
                <a href="#" class="text-gray-400 hover:text-white transition">Privacy</a>
                <a href="#" class="text-gray-400 hover:text-white transition">Terms</a>
                <a href="#" class="text-gray-400 hover:text-white transition">Cookies</a>
            </div>
        </div>
    </div>
</footer>',
                        'dependencies' => ['tailwindcss'],
                    ],
                ],
            ],
            [
                'name' => 'Transparent Navbar',
                'description' => 'Modern navigation bar with transparent background and mobile menu',
                'category_id' => $navbarCategory->id,
                'tags' => ['navigation', 'responsive', 'transparent'],
                'user_id' => $admin->id,
                'status' => 'approved',
                'approved_by' => $admin->id,
                'approved_at' => now(),
                'versions' => [
                    [
                        'framework' => 'react',
                        'styling' => 'tailwind',
                        'code' => 'export default function Navbar() {
    return (
        <nav className="fixed w-full bg-white/90 backdrop-blur-md shadow-sm z-50">
            <div className="container mx-auto px-4">
                <div className="flex justify-between items-center h-16">
                    <div className="text-2xl font-bold text-blue-600">Logo</div>
                    <div className="hidden md:flex space-x-8">
                        <a href="#" className="text-gray-700 hover:text-blue-600 transition">Home</a>
                        <a href="#" className="text-gray-700 hover:text-blue-600 transition">Features</a>
                        <a href="#" className="text-gray-700 hover:text-blue-600 transition">Pricing</a>
                        <a href="#" className="text-gray-700 hover:text-blue-600 transition">Contact</a>
                    </div>
                    <button className="bg-blue-600 text-white px-6 py-2 rounded-lg hover:bg-blue-700 transition">Sign Up</button>
                </div>
            </div>
        </nav>
    );
}',
                        'dependencies' => ['react', 'tailwindcss'],
                    ],
                ],
            ],
            [
                'name' => 'CTA Banner',
                'description' => 'Compelling call-to-action banner with gradient background',
                'category_id' => $ctaCategory->id,
                'tags' => ['cta', 'banner', 'gradient'],
                'user_id' => $contributor->id,
                'status' => 'approved',
                'approved_by' => $admin->id,
                'approved_at' => now(),
                'versions' => [
                    [
                        'framework' => 'html',
                        'styling' => 'tailwind',
                        'code' => '<section class="py-20 bg-gradient-to-r from-blue-600 to-purple-600">
    <div class="container mx-auto px-4 text-center text-white">
        <h2 class="text-5xl font-bold mb-6">Ready to Get Started?</h2>
        <p class="text-2xl mb-10 max-w-2xl mx-auto">Join thousands of satisfied customers and transform your business today</p>
        <div class="flex gap-4 justify-center">
            <button class="bg-white text-blue-600 px-10 py-4 rounded-full text-lg font-semibold hover:bg-gray-100 transition transform hover:scale-105">Start Free Trial</button>
            <button class="border-2 border-white px-10 py-4 rounded-full text-lg font-semibold hover:bg-white hover:text-blue-600 transition">Contact Sales</button>
        </div>
    </div>
</section>',
                        'dependencies' => ['tailwindcss'],
                    ],
                ],
            ],
            [
                'name' => 'Feature Showcase',
                'description' => 'Feature section with alternating image and text layout',
                'category_id' => $featuresCategory->id,
                'tags' => ['features', 'showcase', 'images'],
                'user_id' => $admin->id,
                'status' => 'approved',
                'approved_by' => $admin->id,
                'approved_at' => now(),
                'versions' => [
                    [
                        'framework' => 'html',
                        'styling' => 'tailwind',
                        'code' => '<section class="py-20">
    <div class="container mx-auto px-4">
        <div class="grid grid-cols-1 md:grid-cols-2 gap-12 items-center mb-20">
            <div>
                <h3 class="text-4xl font-bold mb-6">Powerful Analytics</h3>
                <p class="text-xl text-gray-600 mb-6">Track your performance with real-time analytics and detailed insights</p>
                <ul class="space-y-3">
                    <li class="flex items-center"><span class="text-green-500 mr-3 text-xl">✓</span>Real-time data</li>
                    <li class="flex items-center"><span class="text-green-500 mr-3 text-xl">✓</span>Custom reports</li>
                    <li class="flex items-center"><span class="text-green-500 mr-3 text-xl">✓</span>Export capabilities</li>
                </ul>
            </div>
            <div class="bg-gradient-to-br from-blue-100 to-purple-100 rounded-2xl h-96 flex items-center justify-center">
                <span class="text-6xl">📊</span>
            </div>
        </div>
        <div class="grid grid-cols-1 md:grid-cols-2 gap-12 items-center">
            <div class="bg-gradient-to-br from-green-100 to-blue-100 rounded-2xl h-96 flex items-center justify-center order-2 md:order-1">
                <span class="text-6xl">🔒</span>
            </div>
            <div class="order-1 md:order-2">
                <h3 class="text-4xl font-bold mb-6">Enterprise Security</h3>
                <p class="text-xl text-gray-600 mb-6">Your data is protected with bank-level security measures</p>
                <ul class="space-y-3">
                    <li class="flex items-center"><span class="text-green-500 mr-3 text-xl">✓</span>256-bit encryption</li>
                    <li class="flex items-center"><span class="text-green-500 mr-3 text-xl">✓</span>SOC 2 compliant</li>
                    <li class="flex items-center"><span class="text-green-500 mr-3 text-xl">✓</span>Regular audits</li>
                </ul>
            </div>
        </div>
    </div>
</section>',
                        'dependencies' => ['tailwindcss'],
                    ],
                ],
            ],
            [
                'name' => 'Team Cards',
                'description' => 'Team member cards with social links and hover effects',
                'category_id' => $teamCategory->id,
                'tags' => ['team', 'cards', 'social'],
                'user_id' => $contributor->id,
                'status' => 'approved',
                'approved_by' => $admin->id,
                'approved_at' => now(),
                'versions' => [
                    [
                        'framework' => 'react',
                        'styling' => 'tailwind',
                        'code' => 'export default function TeamCards() {
    const team = [
        { name: "Alex Morgan", role: "CEO & Founder", image: "team1.jpg", social: { twitter: "#", linkedin: "#" } },
        { name: "Sarah Chen", role: "CTO", image: "team2.jpg", social: { twitter: "#", linkedin: "#" } },
        { name: "Mike Johnson", role: "Head of Design", image: "team3.jpg", social: { twitter: "#", linkedin: "#" } },
    ];

    return (
        <section className="py-20 bg-gray-50">
            <div className="container mx-auto px-4">
                <h2 className="text-5xl font-bold text-center mb-4">Meet Our Team</h2>
                <p className="text-xl text-gray-600 text-center mb-16">The talented people behind our success</p>
                <div className="grid grid-cols-1 md:grid-cols-3 gap-8">
                    {team.map((member, index) => (
                        <div key={index} className="bg-white rounded-2xl overflow-hidden shadow-lg hover:shadow-2xl transition-all duration-300 group">
                            <div className="h-64 bg-gradient-to-br from-blue-400 to-purple-500 flex items-center justify-center text-6xl">
                                👤
                            </div>
                            <div className="p-6 text-center">
                                <h3 className="text-2xl font-bold mb-2">{member.name}</h3>
                                <p className="text-gray-600 mb-4">{member.role}</p>
                                <div className="flex justify-center gap-4">
                                    <a href={member.social.twitter} className="text-gray-400 hover:text-blue-500 transition">Twitter</a>
                                    <a href={member.social.linkedin} className="text-gray-400 hover:text-blue-700 transition">LinkedIn</a>
                                </div>
                            </div>
                        </div>
                    ))}
                </div>
            </div>
        </section>
    );
}',
                        'dependencies' => ['react', 'tailwindcss'],
                    ],
                ],
            ],
            [
                'name' => 'Contact Card',
                'description' => 'Contact information card with map and details',
                'category_id' => $contactCategory->id,
                'tags' => ['contact', 'map', 'info'],
                'user_id' => $admin->id,
                'status' => 'approved',
                'approved_by' => $admin->id,
                'approved_at' => now(),
                'versions' => [
                    [
                        'framework' => 'html',
                        'styling' => 'tailwind',
                        'code' => '<section class="py-20 bg-gray-50">
    <div class="container mx-auto px-4">
        <h2 class="text-5xl font-bold text-center mb-16">Get In Touch</h2>
        <div class="grid grid-cols-1 md:grid-cols-2 gap-8 max-w-6xl mx-auto">
            <div class="bg-white rounded-2xl shadow-xl p-8">
                <h3 class="text-3xl font-bold mb-6">Contact Information</h3>
                <div class="space-y-6">
                    <div class="flex items-start">
                        <span class="text-3xl mr-4">📍</span>
                        <div>
                            <h4 class="font-semibold text-lg">Address</h4>
                            <p class="text-gray-600">123 Business St, Suite 100<br>New York, NY 10001</p>
                        </div>
                    </div>
                    <div class="flex items-start">
                        <span class="text-3xl mr-4">📧</span>
                        <div>
                            <h4 class="font-semibold text-lg">Email</h4>
                            <p class="text-gray-600">hello@company.com</p>
                        </div>
                    </div>
                    <div class="flex items-start">
                        <span class="text-3xl mr-4">📞</span>
                        <div>
                            <h4 class="font-semibold text-lg">Phone</h4>
                            <p class="text-gray-600">+1 (555) 123-4567</p>
                        </div>
                    </div>
                </div>
            </div>
            <div class="bg-gray-200 rounded-2xl h-96 flex items-center justify-center">
                <span class="text-6xl">🗺️</span>
            </div>
        </div>
    </div>
</section>',
                        'dependencies' => ['tailwindcss'],
                    ],
                ],
            ],
            [
                'name' => 'Stats Counter',
                'description' => 'Animated statistics counter section',
                'category_id' => $aboutCategory->id,
                'tags' => ['stats', 'counter', 'numbers'],
                'user_id' => $contributor->id,
                'status' => 'approved',
                'approved_by' => $admin->id,
                'approved_at' => now(),
                'versions' => [
                    [
                        'framework' => 'html',
                        'styling' => 'tailwind',
                        'code' => '<section class="py-20 bg-blue-600 text-white">
    <div class="container mx-auto px-4">
        <div class="grid grid-cols-1 md:grid-cols-4 gap-8 text-center">
            <div>
                <div class="text-6xl font-bold mb-2">10K+</div>
                <div class="text-xl">Happy Customers</div>
            </div>
            <div>
                <div class="text-6xl font-bold mb-2">50+</div>
                <div class="text-xl">Countries</div>
            </div>
            <div>
                <div class="text-6xl font-bold mb-2">99.9%</div>
                <div class="text-xl">Uptime</div>
            </div>
            <div>
                <div class="text-6xl font-bold mb-2">24/7</div>
                <div class="text-xl">Support</div>
            </div>
        </div>
    </div>
</section>',
                        'dependencies' => ['tailwindcss'],
                    ],
                ],
            ],
            [
                'name' => 'Blog Featured Post',
                'description' => 'Large featured blog post with image and excerpt',
                'category_id' => $blogsCategory->id,
                'tags' => ['featured', 'blog', 'hero'],
                'user_id' => $admin->id,
                'status' => 'approved',
                'approved_by' => $admin->id,
                'approved_at' => now(),
                'versions' => [
                    [
                        'framework' => 'html',
                        'styling' => 'tailwind',
                        'code' => '<section class="py-20">
    <div class="container mx-auto px-4">
        <div class="max-w-6xl mx-auto bg-white rounded-2xl shadow-2xl overflow-hidden">
            <div class="grid grid-cols-1 md:grid-cols-2">
                <div class="h-96 bg-gradient-to-br from-purple-400 to-pink-500 flex items-center justify-center text-8xl">
                    📝
                </div>
                <div class="p-12 flex flex-col justify-center">
                    <span class="text-blue-600 font-semibold mb-2">FEATURED POST</span>
                    <h2 class="text-4xl font-bold mb-4">The Future of Web Development</h2>
                    <p class="text-gray-600 text-lg mb-6">Discover the latest trends and technologies shaping the future of web development in 2024 and beyond.</p>
                    <div class="flex items-center justify-between">
                        <span class="text-gray-500">Dec 22, 2024</span>
                        <button class="bg-blue-600 text-white px-6 py-3 rounded-lg hover:bg-blue-700 transition">Read More</button>
                    </div>
                </div>
            </div>
        </div>
    </div>
</section>',
                        'dependencies' => ['tailwindcss'],
                    ],
                ],
            ],
            [
                'name' => 'Testimonial Grid',
                'description' => 'Grid layout of customer testimonials with ratings',
                'category_id' => $testimonialsCategory->id,
                'tags' => ['testimonials', 'grid', 'ratings'],
                'user_id' => $contributor->id,
                'status' => 'approved',
                'approved_by' => $admin->id,
                'approved_at' => now(),
                'versions' => [
                    [
                        'framework' => 'react',
                        'styling' => 'tailwind',
                        'code' => 'export default function TestimonialGrid() {
    const testimonials = [
        { name: "Emily Davis", role: "Marketing Director", text: "Absolutely game-changing! Our productivity increased by 200%.", rating: 5 },
        { name: "James Wilson", role: "Startup Founder", text: "Best investment we made this year. Highly recommend!", rating: 5 },
        { name: "Lisa Anderson", role: "Product Manager", text: "Intuitive, powerful, and reliable. Everything we needed.", rating: 5 },
    ];

    return (
        <section className="py-20 bg-gray-50">
            <div className="container mx-auto px-4">
                <h2 className="text-5xl font-bold text-center mb-16">Loved by Thousands</h2>
                <div className="grid grid-cols-1 md:grid-cols-3 gap-8">
                    {testimonials.map((testimonial, index) => (
                        <div key={index} className="bg-white p-8 rounded-xl shadow-lg">
                            <div className="flex text-yellow-400 mb-4">
                                {[...Array(testimonial.rating)].map((_, i) => (
                                    <span key={i} className="text-2xl">★</span>
                                ))}
                            </div>
                            <p className="text-gray-700 text-lg mb-6 italic">"{testimonial.text}"</p>
                            <div>
                                <h4 className="font-bold text-lg">{testimonial.name}</h4>
                                <p className="text-gray-600">{testimonial.role}</p>
                            </div>
                        </div>
                    ))}
                </div>
            </div>
        </section>
    );
}',
                        'dependencies' => ['react', 'tailwindcss'],
                    ],
                ],
            ],
        ];

        // Create sections with versions
        foreach ($sectionsData as $sectionData) {
            $versions = $sectionData['versions'];
            unset($sectionData['versions']);

            $section = Section::create($sectionData);

            foreach ($versions as $versionData) {
                $section->versions()->create($versionData);
            }
        }
    }
}
