<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Str;

class Category extends Model
{
    use HasFactory;

    protected $fillable = [
        'user_id',
        'name',
        'slug',
        'type',
        'description',
        'icon',
        'color',
    ];

    /**
     * Boot the model.
     */
    protected static function boot()
    {
        parent::boot();

        static::creating(function ($category) {
            if (empty($category->slug)) {
                $category->slug = Str::slug($category->name);
            }
        });
    }

    /**
     * Get the user who created this category.
     */
    public function user()
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Get sections in this category.
     */
    public function sections()
    {
        return $this->hasMany(Section::class);
    }

    /**
     * Scope to filter categories by user access type.
     */
    public function scopeByAccessType($query, $user)
    {
        // Admins can see all categories
        if ($user->isAdmin()) {
            return $query;
        }

        // If access_type is null or 'both', show all categories
        if (!$user->access_type || $user->access_type === 'both') {
            return $query;
        }

        // Filter based on contributor access type (web or dashboard)
        return $query->where('type', $user->access_type);
    }

    /**
     * Get approved sections count.
     */
    public function approvedSectionsCount()
    {
        return $this->sections()->where('status', 'approved')->count();
    }
}
