<?php

namespace App\Http\Controllers;

use App\Models\Category;
use App\Models\Section;
use App\Models\SectionVersion;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Inertia\Inertia;

class SectionController extends Controller
{
    /**
     * Display a listing of sections
     */
    public function index(Request $request)
    {
        $query = Section::with(['user', 'versions', 'category'])
            ->where('status', 'approved');

        // Filter by category
        if ($request->filled('category')) {
            $query->where('category_id', $request->category);
        }

        // Search by name or description
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                    ->orWhere('description', 'like', "%{$search}%");
            });
        }

        // Filter by tags
        if ($request->filled('tags')) {
            $tags = explode(',', $request->tags);
            foreach ($tags as $tag) {
                $query->whereJsonContains('tags', trim($tag));
            }
        }

        // Sort
        $sortBy = $request->get('sort', 'latest');
        match ($sortBy) {
            'popular' => $query->orderBy('views_count', 'desc'),
            'most_copied' => $query->orderBy('copies_count', 'desc'),
            'oldest' => $query->orderBy('created_at', 'asc'),
            default => $query->orderBy('created_at', 'desc'),
        };

        // Filter sections by category type based on user access
        if (auth('web')->check()) {
            $user = auth('web')->user();
            // Only filter for non-admin users
            if (!$user->isAdmin() && $user->access_type !== 'both') {
                $query->whereHas('category', function ($q) use ($user) {
                    $q->where('type', $user->access_type);
                });
            }
        }

        $sections = $query->paginate($request->get('per_page', 12));

        // Get available categories based on user access
        $categoriesQuery = Category::orderBy('name');
        if (auth('web')->check()) {
            $categoriesQuery->byAccessType(auth('web')->user());
        }
        $categories = $categoriesQuery->get();

        return Inertia::render('sections/index', [
            'sections' => $sections,
            'categories' => $categories,
            'filters' => [
                'category' => $request->category,
                'search' => $request->search,
                'tags' => $request->tags,
                'sort' => $sortBy,
            ],
        ]);
    }

    /**
     * Display a single section with all its versions
     */
    public function show(string $slug)
    {
        $query = Section::with(['user', 'category', 'versions' => function ($query) {
            $query->where('is_active', true);
        }])->where('slug', $slug);

        // Admins can view all sections
        if (auth('web')->check() && auth('web')->user()->isAdmin()) {
            // No status restriction for admins
        } elseif (auth('web')->check()) {
            // Regular users can view approved sections or their own sections
            $query->where(function ($q) {
                $q->where('status', 'approved')
                    ->orWhere('user_id', auth('web')->id());
            });
        } else {
            // Public users can only see approved sections
            $query->where('status', 'approved');
        }

        $section = $query->firstOrFail();

        // Only increment views for approved sections
        if ($section->status === 'approved') {
            $section->incrementViews();
            // Reload the section to get the updated views_count
            $section->refresh();
        }

        return Inertia::render('sections/show', [
            'section' => $section,
        ]);
    }

    /**
     * Get code for a specific version
     */
    public function getCode(Section $section, SectionVersion $version)
    {
        if ($section->status !== 'approved') {
            abort(404);
        }

        if ($version->section_id !== $section->id) {
            abort(404);
        }

        // Increment copies count
        $section->incrementCopies();
        // Reload to get the updated copies_count
        $section->refresh();

        return response()->json([
            'code' => $version->code,
            'framework' => $version->framework,
            'styling' => $version->styling,
            'dependencies' => $version->dependencies,
            'copies_count' => $section->copies_count,
        ]);
    }

    /**
     * Show the form for creating a new section
     */
    public function create()
    {
        $categories = Category::byAccessType(request()->user())
            ->orderBy('name')
            ->get();

        return Inertia::render('sections/create', [
            'categories' => $categories,
        ]);
    }

    /**
     * Store a new section (for contributors)
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'category_id' => 'required|exists:categories,id',
            'tags' => 'nullable|array',
            'tags.*' => 'string',
            'thumbnail' => 'nullable|image|max:2048',
            'versions' => 'required|array|min:1',
            'versions.*.framework' => 'required|in:html,react,vue,angular',
            'versions.*.styling' => 'required|in:tailwind,bootstrap',
            'versions.*.code' => 'required|string',
            'versions.*.dependencies' => 'nullable|array',
            'versions.*.notes' => 'nullable|string',
        ]);

        // Check for duplicate frameworks
        $frameworks = collect($validated['versions'])->pluck('framework')->toArray();
        $uniqueFrameworks = array_unique($frameworks);

        if (count($frameworks) !== count($uniqueFrameworks)) {
            return back()->withErrors([
                'versions' => 'You can only add one version per framework. Please ensure each framework (HTML, React, Vue, Angular) appears only once.'
            ])->withInput();
        }

        // Handle thumbnail upload
        $thumbnailPath = null;
        if ($request->hasFile('thumbnail')) {
            $thumbnailPath = $request->file('thumbnail')->store('sections/thumbnails', 'public');
        }

        // Determine status based on user role
        $status = $request->user()->isAdmin() ? 'approved' : 'pending';

        try {
            // Create section
            $section = Section::create([
                'name' => $validated['name'],
                'description' => $validated['description'] ?? null,
                'category_id' => $validated['category_id'],
                'tags' => $validated['tags'] ?? [],
                'thumbnail' => $thumbnailPath,
                'status' => $status,
                'user_id' => $request->user()->id,
                'approved_by' => $status === 'approved' ? $request->user()->id : null,
                'approved_at' => $status === 'approved' ? now() : null,
            ]);

            // Create versions
            foreach ($validated['versions'] as $versionData) {
                $section->versions()->create([
                    'framework' => $versionData['framework'],
                    'styling' => $versionData['styling'],
                    'code' => $versionData['code'],
                    'dependencies' => $versionData['dependencies'] ?? [],
                    'notes' => $versionData['notes'] ?? null,
                ]);
            }

            return redirect()->route('my-sections.index')
                ->with('success', $status === 'approved'
                    ? 'Section created and published successfully!'
                    : 'Section submitted for approval!');
        } catch (\Exception $e) {
            return back()
                ->with('error', 'An error occurred while creating the section. Please try again.')
                ->withInput();
        }
    }

    /**
     * Show the form for editing a section
     */
    public function edit(Section $section)
    {
        // Authorization
        if ($section->user_id !== request()->user()->id && ! request()->user()->isAdmin()) {
            abort(403, 'Unauthorized');
        }

        $categories = Category::byAccessType(request()->user())
            ->orderBy('name')
            ->get();

        return Inertia::render('sections/edit', [
            'section' => $section->load(['versions', 'category']),
            'categories' => $categories,
        ]);
    }

    /**
     * Update a section (only by creator or admin)
     */
    public function update(Request $request, Section $section)
    {
        // Authorization
        if ($section->user_id !== $request->user()->id && ! $request->user()->isAdmin()) {
            abort(403, 'Unauthorized');
        }

        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'category_id' => 'required|exists:categories,id',
            'tags' => 'nullable|array',
            'tags.*' => 'string',
            'thumbnail' => 'nullable|image|max:2048',
            'versions' => 'required|array|min:1',
            'versions.*.id' => 'nullable|exists:section_versions,id',
            'versions.*.framework' => 'required|in:html,react,vue,angular',
            'versions.*.styling' => 'required|in:tailwind,bootstrap',
            'versions.*.code' => 'required|string',
            'versions.*.dependencies' => 'nullable|array',
            'versions.*.notes' => 'nullable|string',
        ]);

        // Check for duplicate frameworks
        $frameworks = collect($validated['versions'])->pluck('framework')->toArray();
        $uniqueFrameworks = array_unique($frameworks);

        if (count($frameworks) !== count($uniqueFrameworks)) {
            return back()->withErrors([
                'versions' => 'You can only add one version per framework. Please ensure each framework (HTML, React, Vue, Angular) appears only once.'
            ])->withInput();
        }

        // Handle thumbnail upload
        if ($request->hasFile('thumbnail')) {
            // Delete old thumbnail if it exists
            if ($section->thumbnail) {
                Storage::disk('public')->delete($section->thumbnail);
            }
            $validated['thumbnail'] = $request->file('thumbnail')->store('sections/thumbnails', 'public');
        }

        // If contributor updates an approved section, reset to pending
        if (! $request->user()->isAdmin() && $section->status === 'approved') {
            $validated['status'] = 'pending';
            $validated['approved_by'] = null;
            $validated['approved_at'] = null;
        }

        try {
            // Update section
            $section->update([
                'name' => $validated['name'],
                'description' => $validated['description'] ?? null,
                'category_id' => $validated['category_id'],
                'tags' => $validated['tags'] ?? [],
                'thumbnail' => $validated['thumbnail'] ?? $section->thumbnail,
                'status' => $validated['status'] ?? $section->status,
                'approved_by' => $validated['approved_by'] ?? $section->approved_by,
                'approved_at' => $validated['approved_at'] ?? $section->approved_at,
            ]);

            // Handle versions - delete old ones and create new ones
            $section->versions()->delete();

            foreach ($validated['versions'] as $versionData) {
                $section->versions()->create([
                    'framework' => $versionData['framework'],
                    'styling' => $versionData['styling'],
                    'code' => $versionData['code'],
                    'dependencies' => $versionData['dependencies'] ?? [],
                    'notes' => $versionData['notes'] ?? null,
                    'is_active' => true,
                ]);
            }

            return redirect()->route('my-sections.index')
                ->with('success', 'Section updated successfully!');
        } catch (\Exception $e) {
            return back()
                ->with('error', 'An error occurred while updating the section. Please try again.')
                ->withInput();
        }
    }

    /**
     * Resubmit a rejected section for approval
     */
    public function resubmit(Request $request, Section $section)
    {
        // Authorization - only section owner can resubmit
        if ($section->user_id !== $request->user()->id) {
            abort(403, 'Unauthorized');
        }

        // Only rejected sections can be resubmitted
        if ($section->status !== 'rejected') {
            return back()->with('error', 'Only rejected sections can be resubmitted');
        }

        // Update status to pending
        $section->update([
            'status' => 'pending',
            'rejection_reason' => null,
            'approved_by' => null,
            'approved_at' => null,
        ]);

        return back()->with('success', 'Section resubmitted for approval!');
    }

    /**
     * Delete a section (only by creator or admin)
     */
    public function destroy(Request $request, Section $section)
    {
        // Authorization
        if ($section->user_id !== $request->user()->id && ! $request->user()->isAdmin()) {
            abort(403, 'Unauthorized');
        }

        $section->delete();

        return back()->with('success', 'Section deleted successfully!');
    }
}
