<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Category;
use Illuminate\Http\Request;
use Inertia\Inertia;

class CategoryController extends Controller
{
    /**
     * Display a listing of the categories.
     */
    public function index(Request $request)
    {
        $user = $request->user();

        $query = Category::query();

        // Non-admins can only see their own categories
        if (!$user->isAdmin()) {
            $query->where('user_id', $user->id);
        }

        // Filter by access type
        $query->byAccessType($user);

        $categories = $query
            ->withCount(['sections' => function ($query) {
                $query->where('status', 'approved');
            }])
            ->orderBy('name')
            ->paginate(20);

        return Inertia::render('admin/categories/index', [
            'categories' => $categories,
        ]);
    }

    /**
     * Show the form for creating a new category.
     */
    public function create(Request $request)
    {
        $accessibleTypes = $request->user()->getAccessibleCategoryTypes();

        return Inertia::render('admin/categories/create', [
            'accessibleTypes' => $accessibleTypes,
        ]);
    }

    /**
     * Store a newly created category in storage.
     */
    public function store(Request $request)
    {
        $user = $request->user();
        $accessibleTypes = $user->getAccessibleCategoryTypes();

        $validated = $request->validate([
            'name' => ['required', 'string', 'max:255'],
            'type' => ['required', 'in:web,dashboard'],
            'description' => ['nullable', 'string'],
            'icon' => ['nullable', 'string', 'max:255'],
            'color' => ['nullable', 'string', 'max:7'],
        ]);

        // Check if user has access to create this category type
        if (!in_array($validated['type'], $accessibleTypes)) {
            return back()
                ->withErrors(['type' => 'You do not have permission to create this category type.'])
                ->withInput();
        }

        // Add user_id to the validated data
        $validated['user_id'] = $user->id;

        Category::create($validated);

        return redirect()->route('admin.categories.index')
            ->with('success', 'Category created successfully!');
    }

    /**
     * Show the form for editing the specified category.
     */
    public function edit(Request $request, Category $category)
    {
        $user = $request->user();

        // Check ownership - non-admins can only edit their own categories
        if (!$user->isAdmin() && $category->user_id !== $user->id) {
            return redirect()->route('admin.categories.index')
                ->with('error', 'You do not have permission to edit this category.');
        }

        $accessibleTypes = $user->getAccessibleCategoryTypes();

        return Inertia::render('admin/categories/edit', [
            'category' => $category,
            'accessibleTypes' => $accessibleTypes,
        ]);
    }

    /**
     * Update the specified category in storage.
     */
    public function update(Request $request, Category $category)
    {
        $user = $request->user();

        // Check ownership - non-admins can only update their own categories
        if (!$user->isAdmin() && $category->user_id !== $user->id) {
            return redirect()->route('admin.categories.index')
                ->with('error', 'You do not have permission to update this category.');
        }

        $accessibleTypes = $user->getAccessibleCategoryTypes();

        $validated = $request->validate([
            'name' => ['required', 'string', 'max:255'],
            'type' => ['required', 'in:web,dashboard'],
            'description' => ['nullable', 'string'],
            'icon' => ['nullable', 'string', 'max:255'],
            'color' => ['nullable', 'string', 'max:7'],
        ]);

        // Check if user has access to update to this category type
        if (!in_array($validated['type'], $accessibleTypes)) {
            return back()
                ->withErrors(['type' => 'You do not have permission to set this category type.'])
                ->withInput();
        }

        $category->update($validated);

        return redirect()->route('admin.categories.index')
            ->with('success', 'Category updated successfully!');
    }

    /**
     * Remove the specified category from storage.
     */
    public function destroy(Request $request, Category $category)
    {
        $user = $request->user();

        // Check ownership - non-admins can only delete their own categories
        if (!$user->isAdmin() && $category->user_id !== $user->id) {
            return redirect()->route('admin.categories.index')
                ->with('error', 'You do not have permission to delete this category.');
        }

        // Check if category has sections
        if ($category->sections()->count() > 0) {
            return back()->with('error', 'Cannot delete category with existing sections.');
        }

        $category->delete();

        return redirect()->route('admin.categories.index')
            ->with('success', 'Category deleted successfully!');
    }
}
